/************************************************************************
* FILE          : SDS_tclXmlParser.h                                   *
* SW-COMPONENT  : SDS_helpclasses                                      *
* DESCRIPTION   : This is the xml parser ported to OSAL compliance     *
*                 dynamic context handling                             *
* COPYRIGHT     : (c) 2011 RBEI, Bangalore                             *
*                                                                      *
* HISTORY:                                                             *
*                                                                      *
*  DD.MM.YYYY | Version | Author                    | Modification     *
*  --------------------------------------------------------------------*
*  31.01.2011 |   1.0   | RBEI/ECF1 - Jose Antony   | Ported to OSAL   *
************************************************************************/


/**
 ****************************************************************************
 * <P> XML.c - implementation file for basic XML parser written in ANSI C++
 * for portability. It works by using recursion and a node tree for breaking
 * down the elements of an XML document.  </P>
 *
 * @version     V2.23
 * @author      Frank Vanden Berghen
 *
 * BSD license:
 * Copyright (c) 2002, Frank Vanden Berghen
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the Frank Vanden Berghen nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/


#ifndef _FC_RECOG_TCLXMLPARSER_H_
#define _FC_RECOG_TCLXMLPARSER_H_

#define XMLCSTR tCString
#define XMLSTR  tString
#define XMLCHAR tChar

 
#ifndef FALSE
    #define FALSE 0
#endif /* FALSE */
#ifndef TRUE
    #define TRUE 1
#endif /* TRUE */


// Enumeration for XML parse errors.
typedef enum XMLError
{
    eXMLErrorNone = 0,
    eXMLErrorMissingEndTag,
    eXMLErrorEmpty,
    eXMLErrorFirstNotStartTag,
    eXMLErrorMissingTagName,
    eXMLErrorMissingEndTagName,
    eXMLErrorNoMatchingQuote,
    eXMLErrorUnmatchedEndTag,
    eXMLErrorUnmatchedEndClearTag,
    eXMLErrorUnexpectedToken,
    eXMLErrorInvalidTag,
    eXMLErrorNoElements,
    eXMLErrorFileNotFound,
    eXMLErrorFirstTagNotFound,
    eXMLErrorUnknownCharacterEntity,
    eXMLErrorCharConversionError,
    eXMLErrorCannotOpenWriteFile,
    eXMLErrorCannotWriteFile
} XMLError;



// Enumeration used to manage type of data. Use in conjunction with structure XMLNodeContents
typedef enum XMLElementType
{
    eNodeChild=0,
    eNodeAttribute=1,
    eNodeText=2,
    eNodeClear=3,
    eNodeNULL=4
} XMLElementType;



// Structure used to obtain error details if the parse fails.
typedef struct XMLResults
{
    enum XMLError error;
    tS32  nLine,nColumn;
} XMLResults;



// Structure for XML clear (unformatted) node (usually comments)
typedef struct {
    XMLCSTR lpszValue; XMLCSTR lpszOpenTag; XMLCSTR lpszCloseTag;
} XMLClear;


// Structure for XML attribute.
typedef struct {
    XMLCSTR lpszName; XMLCSTR lpszValue;
} XMLAttribute;


// Structure for XML clear tags.
typedef struct {
    XMLCSTR lpszOpen; tS32 openTagLen; XMLCSTR lpszClose;
} ALLXMLClearTag;


struct XMLNodeContents;

typedef struct XMLNode
{
  private:

    struct XMLNodeDataTag;


    // protected constructors: use one of these four methods to get your first instance of XMLNode:
    //  - parseString
    //  - parseFile
    //  - openFileHelper
    //  - createXMLTopNode
    XMLNode(struct XMLNodeDataTag *pParent, XMLCSTR lpszName, tChar isDeclaration);
    XMLNode(struct XMLNodeDataTag *p);

  public:

      // You can create your first instance of XMLNode with these 4 functions:
      // (see complete explanation of parameters below)

    
	static XMLNode createXMLTopNode(XMLCSTR lpszName, tChar isDeclaration=FALSE);
    static XMLNode parseString   (XMLCSTR  lpXMLString, XMLCSTR tag=OSAL_NULL, XMLResults *pResults=OSAL_NULL);
    static XMLNode parseFile     (XMLCSTR     filename, XMLCSTR tag=OSAL_NULL, XMLResults *pResults=OSAL_NULL);
    static XMLNode openFileHelper(XMLCSTR     filename, XMLCSTR tag=OSAL_NULL                           );

    // The tag parameter should be the name of the first tag inside the XML file.
    // If the tag parameter is omitted, the 3 functions return a node that represents
    // the head of the xml document including the declaration term (<? ... ?>).

    // The "openFileHelper" reports to the screen all the warnings & errors that occurred during
    // parsing of the XML file. Since each application has its own way to report and deal with errors,
    // you should rather use the "parseFile" function to parse XML files and program yourself thereafter
    // an "error reporting" tailored for your needs (instead of using the very crude "error reporting"
    // mechanism included inside the "openFileHelper" function).

    // If the XML document is corrupted:
    //   * The "openFileHelper" method will:
    //         - display an error message on the console (or inside a messageBox for windows).
    //         - stop execution (exit).
    //     I suggest that you write your own "openFileHelper" method tailored to your needs.
    //   * The 2 other methods will initialize the "pResults" variable with some information that
    //     can be used to trace the error.
    //   * If you still want to parse the file, you can use the APPROXIMATE_PARSING option as
    //     explained inside the note at the beginning of the "xmlParser.cpp" file.
    // You can have a user-friendly explanation of the parsing error with this function:
    static XMLCSTR getError(XMLError error);
    static XMLCSTR getVersion();
    static ALLXMLClearTag* getClearTagTable();

    XMLCSTR getName() const;                                         // name of the node
    XMLCSTR getText(tS32 i=0) const;                                  // return ith text field
    tS32 nText() const;                                               // nbr of text field
    XMLNode getParentNode() const;                                   // return the parent node
    XMLNode getChildNode(tS32 i=0) const;                             // return ith child node
    XMLNode getChildNode(XMLCSTR name, tS32 i)  const;                // return ith child node with specific name
                                                                     //     (return an empty node if failing)
    XMLNode getChildNode(XMLCSTR name, tPS32 i=OSAL_NULL) const;           // return next child node with specific name
                                                                     //     (return an empty node if failing)
    XMLNode getChildNodeWithAttribute(XMLCSTR tagName,               // return child node with specific name/attribute
                                      XMLCSTR attributeName,         //     (return an empty node if failing)
                                      XMLCSTR attributeValue=OSAL_NULL,   //
                                      tPS32 i=OSAL_NULL)  const;           //


    tS32 nChildNode(XMLCSTR name) const;                              // return the number of child node with specific name
    tS32 nChildNode() const;                                          // nbr of child node
    XMLAttribute getAttribute(tS32 i=0) const;                        // return ith attribute
    XMLCSTR      getAttributeName(tS32 i=0) const;                    // return ith attribute name
    XMLCSTR      getAttributeValue(tS32 i=0) const;                   // return ith attribute value
    tChar  isAttributeSet(XMLCSTR name) const;                        // test if an attribute with a specific name is given
    XMLCSTR getAttribute(XMLCSTR name, tS32 i) const;                 // return ith attribute content with specific name
                                                                     //     (return a NULL if failing)
    XMLCSTR getAttribute(XMLCSTR name, tPS32 i=OSAL_NULL) const;           // return next attribute content with specific name
                                                                     //     (return a NULL if failing)
    tS32 nAttribute() const;                                          // nbr of attribute
    XMLClear getClear(tS32 i=0) const;                                // return ith clear field (comments)
    tS32 nClear() const;    	// nbr of clear field
	
	XMLSTR createXMLString(tS32 nFormat=1, tPS32 pnSize=OSAL_NULL) const;   // create XML string starting from current XMLNode
                                                                     // if nFormat==0, no formatting is required
                                                                     // otherwise this returns an user friendly XML string from a
                                                                     // given element with appropriate white spaces and carriage returns.
                                                                     // if pnSize is given it returns the size in character of the string.
    XMLError writeToFile(XMLCSTR filename, tCString encoding=OSAL_NULL, tChar nFormat=1) const;
                                                                     // save the content of an xmlNode inside a file.
                                                                     // the nFormat parameter has the same meaning as in the
                                                                     // createXMLString function. If "strictUTF8Parsing=1", the
                                                                     // the encoding parameter is ignored and always set to
                                                                     // "utf-8". If "_XMLUNICODE=1", the encoding parameter is
                                                                     // ignored and always set to "utf-16".

	XMLNodeContents enumContents(tS32 i) const;                       // enumerate all the different contents (attribute,child,text,
                                                                     //     clear) of the current XMLNode. The order is reflecting
                                                                     //     the order of the original file/string.
                                                                     //     NOTE: 0 <= i < nElement();
    tS32 nElement() const;                                            // nbr of different contents for current node
    tChar isEmpty() const;                                            // is this node Empty?
    tChar isDeclarationSet() const;         	// is this node a declaration <? .... ?>
	
	
	// to allow shallow/fast copy:
    ~XMLNode();
    XMLNode(const XMLNode &A);
    XMLNode& operator=( const XMLNode& A );

    XMLNode(): d(OSAL_NULL){};
    static XMLNode emptyXMLNode;
    static XMLClear emptyXMLClear;
    static XMLAttribute emptyXMLAttribute;
	
	
	
	// The following functions allows you to create from scratch (or update) a XMLNode structure
    // Start by creating your top node with the "createXMLTopNode" function and then add new nodes with the "addChild" function.
    // The parameter 'pos' gives the position where the childNode, the text or the XMLClearTag will be inserted.
    // The default value (pos=-1) inserts at the end. The value (pos=0) insert at the beginning (Insertion at the beginning is slower than at the end).
    // REMARK: 0 <= pos < nChild()+nText()+nClear()
    XMLNode       addChild(XMLCSTR lpszName, tChar isDeclaration=FALSE, tS32 pos=-1);
    XMLAttribute *addAttribute(XMLCSTR lpszName, XMLCSTR lpszValuev);
    XMLCSTR       addText(XMLCSTR lpszValue, tS32 pos=-1);
    XMLClear     *addClear(XMLCSTR lpszValue, XMLCSTR lpszOpen=OSAL_NULL, XMLCSTR lpszClose=OSAL_NULL, tS32 pos=-1);
                                                                    // default values: lpszOpen=XMLNode::getClearTagTable()->lpszOpen;
                                                                    //                 lpszClose=XMLNode::getClearTagTable()->lpszClose;
    XMLNode       addChild(XMLNode nodeToAdd, tS32 pos=-1);          // If the "nodeToAdd" has some parents, it will be detached
                                                                    // from it's parents before being attached to the current XMLNode
    // Some update functions:
    XMLCSTR       updateName(XMLCSTR lpszName);                                                    // change node's name
    XMLAttribute *updateAttribute(XMLAttribute *newAttribute, XMLAttribute *oldAttribute);         // if the attribute to update is missing, a new one will be added
    XMLAttribute *updateAttribute(XMLCSTR lpszNewValue, XMLCSTR lpszNewName=OSAL_NULL,tS32 i=0);         // if the attribute to update is missing, a new one will be added
    XMLAttribute *updateAttribute(XMLCSTR lpszNewValue, XMLCSTR lpszNewName,XMLCSTR lpszOldName);  // set lpszNewName=NULL if you don't want to change the name of the attribute
                                                                                                   // if the attribute to update is missing, a new one will be added
    XMLCSTR       updateText(XMLCSTR lpszNewValue, tS32 i=0);                                       // if the text to update is missing, a new one will be added
    XMLCSTR       updateText(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue);                          // if the text to update is missing, a new one will be added
    XMLClear     *updateClear(XMLCSTR lpszNewContent, tS32 i=0);                                    // if the clearTag to update is missing, a new one will be added
    XMLClear     *updateClear(XMLClear *newP,XMLClear *oldP);                                      // if the clearTag to update is missing, a new one will be added
    XMLClear     *updateClear(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue);                         // if the clearTag to update is missing, a new one will be added

	
	 // Some deletion functions:
    /*
    tVoid deleteNodeContent(tChar force=0);  // delete the content of this XMLNode and the subtree.
                                           // if force=0, while (references to this node still exist), no memory free occurs
                                           // if force=1, always delete the content of this XMLNode and the subtree and free associated memory
    */
    /// The "deleteNodeContent" function forces the deletion of the content of this XMLNode and the subtree.
    void deleteNodeContent();

    tVoid deleteAttribute(XMLCSTR lpszName);
    tVoid deleteAttribute(tS32 i=0);
    tVoid deleteAttribute(XMLAttribute *anAttribute);
    tVoid deleteText(tS32 i=0);
    tVoid deleteText(XMLCSTR lpszValue);
    tVoid deleteClear(tS32 i=0);
    tVoid deleteClear(XMLClear *p);
    tVoid deleteClear(XMLCSTR lpszValue);

	
	//some useful add and update functions
	
	static XMLNode createXMLTopNode_WOSD(XMLCSTR lpszName, tChar isDeclaration=FALSE);
    XMLNode        addChild_WOSD(XMLCSTR lpszName, tChar isDeclaration=FALSE, tS32 pos=-1);
    XMLAttribute  *addAttribute_WOSD(XMLCSTR lpszName, XMLCSTR lpszValue);
    XMLCSTR        addText_WOSD(XMLCSTR lpszValue, tS32 pos=-1);
    XMLClear      *addClear_WOSD(XMLCSTR lpszValue, XMLCSTR lpszOpen=OSAL_NULL, XMLCSTR lpszClose=OSAL_NULL, tS32 pos=-1);

    XMLCSTR        updateName_WOSD(XMLCSTR lpszName);
    XMLAttribute  *updateAttribute_WOSD(XMLAttribute *newAttribute, XMLAttribute *oldAttribute);
    XMLAttribute  *updateAttribute_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszNewName=OSAL_NULL,tS32 i=0);
    XMLAttribute  *updateAttribute_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszNewName,XMLCSTR lpszOldName);
    XMLCSTR        updateText_WOSD(XMLCSTR lpszNewValue, tS32 i=0);
    XMLCSTR        updateText_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue);
    XMLClear      *updateClear_WOSD(XMLCSTR lpszNewContent, tS32 i=0);
    XMLClear      *updateClear_WOSD(XMLClear *newP,XMLClear *oldP);
    XMLClear      *updateClear_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue);
	
	// These are some useful functions when you want to insert a childNode, a text or a XMLClearTag in the
    // middle (at a specified position) of a XMLNode tree already constructed. The value returned by these
    // methods is to be used as last parameter (parameter 'pos') of addChild, addText or addClear.
    tS32 positionOfText(tS32 i=0) const;
    tS32 positionOfText(XMLCSTR lpszValue) const;
    tS32 positionOfClear(tS32 i=0) const;
    tS32 positionOfClear(XMLCSTR lpszValue) const;
    tS32 positionOfClear(XMLClear *a) const;
    tS32 positionOfChildNode(tS32 i=0) const;
    tS32 positionOfChildNode(XMLNode x) const;
    tS32 positionOfChildNode(XMLCSTR name, tS32 i=0) const; // return the position of the ith childNode with the specified name
                                                          // if (name==NULL) return the position of the ith childNode
														  
														  
	// The setGlobalOptions function allows you to change two global parameters that affect string&file
    // parsing. First of all, you most-probably will never have to change these 2 global parameters.
    // About the "guessUnicodeChars" parameter:
    //     If "guessUnicodeChars=1" and if this library is compiled in UNICODE mode, then the
    //     "parseFile" and "openFileHelper" functions will test if the file contains ASCII
    //     characters. If this is the case, then the file will be loaded and converted in memory to
    //     UNICODE before being parsed. If "guessUnicodeChars=0", no conversion will
    //     be performed.
    //
    //     If "guessUnicodeChars=1" and if this library is compiled in ASCII/UTF8 mode, then the
    //     "parseFile" and "openFileHelper" functions will test if the file contains UNICODE
    //     characters. If this is the case, then the file will be loaded and converted in memory to
    //     ASCII/UTF8 before being parsed. If "guessUnicodeChars=0", no conversion will
    //     be performed
    //
    //     Sometime, it's useful to set "guessUnicodeChars=0" to disable any conversion
    //     because the test to detect the file-type (ASCII/UTF8 or UNICODE) may fail (rarely).
    //
    // About the "strictUTF8Parsing" parameter:
    //     If "strictUTF8Parsing=0" then we assume that all characters have the same length of 1 byte.
    //     If "strictUTF8Parsing=1" then the characters have different lengths (from 1 byte to 4 bytes)
    //     depending on the content of the first byte of the character.
    // About the "dropWhiteSpace" parameter:
    //

    static tVoid setGlobalOptions(tChar guessUnicodeChars=1, tChar strictUTF8Parsing=1, tChar dropWhiteSpace=1);

    // The next function try to guess if the character encoding is UTF-8. You most-probably will never
    // have to use this function. It then returns the appropriate value of the global parameter
    // "strictUTF8Parsing" described above. The guess is based on the content of a buffer of length
    // "bufLen" bytes that contains the first bytes (minimum 25 bytes; 200 bytes is a good value) of the
    // file to be parsed. The "openFileHelper" function is using this function to automatically compute
    // the value of the "strictUTF8Parsing" global parameter. There are several heuristics used to do the
    // guess. One of the heuristic is based on the "encoding" attribute. The original XML specifications
    // forbids to use this attribute to do the guess but you can still use it if you set
    // "useXMLEncodingAttribute" to 1 (this is the default behavior and the behavior of most parsers).

   // static char guessUTF8ParsingParameterValue(tPVoid buffer, tS32 bufLen, char useXMLEncodingAttribute=1);
	
	
	
	 private:

// these are functions and structures used internally by the XMLNode class (don't bother about them):

      typedef struct XMLNodeDataTag // to allow shallow copy and "intelligent/smart" pointers (automatic delete):
      {
          XMLCSTR                lpszName;        // Element name (=NULL if root)
          tS32                    nChild,          // Number of child nodes
                                 nText,           // Number of text fields
                                 nClear,          // Number of Clear fields (comments)
                                 nAttribute;      // Number of attributes
          tChar                   isDeclaration;   // Whether node is an XML declaration - '<?xml ?>'
          struct XMLNodeDataTag  *pParent;        // Pointer to parent element (=NULL if root)
          XMLNode                *pChild;         // Array of child nodes
          XMLCSTR                *pText;          // Array of text fields
          XMLClear               *pClear;         // Array of clear fields
          XMLAttribute           *pAttribute;     // Array of attributes
          tS32                    *pOrder;         // order of the child_nodes,text_fields,clear_fields
          tS32                    ref_count;       // for garbage collection (smart pointers)
      } XMLNodeData;
      XMLNodeData *d;

      tChar parseClearTag(tPVoid px, ALLXMLClearTag *pa);
      tChar maybeAddTxT(tPVoid pa, XMLCSTR tokenPStr);
      tS32 ParseXMLElement(tPVoid pXML);
      tPVoid addToOrder(tS32 memInc, tPS32 _pos, tS32 nc, tPVoid p, tS32 size, XMLElementType xtype);
      tS32 indexText(XMLCSTR lpszValue) const;
      tS32 indexClear(XMLCSTR lpszValue) const;
      XMLNode addChild_priv(tS32,XMLCSTR,tChar,tS32);
      XMLAttribute *addAttribute_priv(tS32,XMLCSTR,XMLCSTR);
      XMLCSTR addText_priv(tS32,XMLCSTR,tS32);
      XMLClear *addClear_priv(tS32,XMLCSTR,XMLCSTR,XMLCSTR,tS32);
      static inline tS32 findPosition(XMLNodeData *d, tS32 index, XMLElementType xtype);
      static tS32 CreateXMLStringR(XMLNodeData *pEntry, XMLSTR lpszMarker, tS32 nFormat);
      static tS32 removeOrderElement(XMLNodeData *d, XMLElementType t, tS32 index);
      static tVoid exactMemory(XMLNodeData *d);
      static tS32 detachFromParent(XMLNodeData *d);
      void emptyTheNode(char force);
} XMLNode;


// This structure is given by the function "enumContents".
typedef struct XMLNodeContents
{
    // This dictates what's the content of the XMLNodeContent
    enum XMLElementType type;
    // should be an union to access the appropriate data.
    // compiler does not allow union of object with constructor... too bad.
    XMLNode child;
    XMLAttribute attrib;
    XMLCSTR text;
    XMLClear clear;

} XMLNodeContents;



tVoid free_XMLDLL(tPVoid t); // {free(t);}

// Duplicate (copy in a new allocated buffer) the source string. This is
// a very handy function when used with all the "XMLNode::*_WOSD" functions.
// (If (cbData!=0) then cbData is the number of chars to duplicate)
XMLSTR stringDup(XMLCSTR source, tS32 cbData=0);

// The 3 following functions are processing strings so that all the characters
// &,",',<,> are replaced by their XML equivalent: &amp;, &quot;, &apos;, &lt;, &gt;.
// These 3 functions are useful when creating from scratch an XML file using the
// "printf", "fprintf", "cout",... functions. If you are creating from scratch an
// XML file using the provided XMLNode class you cannot use these functions (the
// XMLNode class does the processing job for you during rendering). The second
// function ("toXMLStringFast") allows you to re-use the same output buffer
// for all the conversions so that only a few memory allocations are performed.
// If the output buffer is too small to contain thee resulting string, it will
// be enlarged.
XMLSTR toXMLString(XMLCSTR source);
XMLSTR toXMLStringFast(XMLSTR *destBuffer,tPS32 destSz, XMLCSTR source);

// you should not use this one (there is a possibility of "destination-buffer-overflow"):
XMLSTR toXMLString(XMLSTR dest,XMLCSTR source);

#endif
